/* batch_ScoreAnimations2-TD-64.jsx — 64x32 TD Animation #2
 * - Sets TOUCHDOWN and TOUCHDOWN 2 text to Primary color
 * - Updates Main-Logo-PreComp with team logo
 * - Updates TeamLogo layer with team logo
 * - Sets TeamName to team abbreviation with Smart Secondary layer style
 * - Fits TeamName text to canvas width by adjusting tracking
 */

(function () {
  // --- load common helpers ---
  (function(){
    var COMMON = $.getenv("AE_COMMON_JSX") || "";
    if (COMMON) {
      try { delete $.global.GL; } catch(e) { $.global.GL = undefined; }
      $.evalFile(File(COMMON));
    }
    if (!$.global.GL) throw new Error("Common library not loaded. Set AE_COMMON_JSX to gl_common.jsxinc");
  })();

  var GL = $.global.GL;

  // --- env & defaults ---
  var PROJECT   = GL.env("AE_PROJECT", null);
  var CSV_PATH  = GL.env("AE_CSV", null);
  var COMP_NAME = GL.env("AE_COMP", "Comp-TD-64");

  // Layer names
  var TOUCHDOWN_LAYER    = GL.cleanValue(GL.env("AE_TOUCHDOWN_LAYER", "TOUCH DOWN"));
  var TOUCHDOWN2_LAYER   = GL.cleanValue(GL.env("AE_TOUCHDOWN2_LAYER", "TOUCH DOWN 2"));
  var TEAMNAME_LAYER     = GL.cleanValue(GL.env("AE_TEAMNAME_LAYER", "TeamName"));
  var MAIN_LOGO_PRECOMP  = GL.cleanValue(GL.env("AE_MAIN_LOGO_PRECOMP", "Main-Logo-PreComp"));
  var TEAMLOGO_LAYER     = GL.cleanValue(GL.env("AE_TEAMLOGO_LAYER", "TeamLogo"));

  var LEAGUE    = GL.env("AE_LEAGUE", "");
  var LIMIT_STR = GL.env("AE_LIMIT", "");
  var LIMIT     = (LIMIT_STR && !isNaN(parseInt(LIMIT_STR,10))) ? parseInt(LIMIT_STR,10) : null;

  var OUTDIR    = GL.env("AE_OUTDIR", "");
  var PATH_TPL  = GL.env("AE_PATH_TEMPLATE", "{league}");
  var ANIM_NAME = GL.env("AE_ANIM", "_TD_2_1");
  var RS_TPL    = GL.env("AE_RS_TEMPLATE", "Best Settings");
  var OM_TPL    = GL.env("AE_OM_TEMPLATE", "PNG Sequence");
  var PURGE     = (GL.env("AE_PURGE_BEFORE_RENDER", "1") === "1");
  var NO_RENDER = (GL.env("AE_NO_RENDER", "0") === "1");
  var QUIT_APP  = (GL.env("AE_QUIT", "1") === "1");

  var logoOpts = {
    dir: GL.env("AE_LOGO_DIR", ""),
    tpl: GL.env("AE_LOGO_PATH_TEMPLATE", "{league}/{abbr}"),
    exts: GL.env("AE_LOGO_EXTS", "png,jpg,jpeg,svg,ai,psd")
  };

  // --- start ---
  if (app.beginSuppressDialogs) { try { app.beginSuppressDialogs(); } catch(e) {} }
  app.beginUndoGroup("TD-64 #2 - Batch Render");

  if (!PROJECT) GL.fail("AE_PROJECT env not set.");
  var aep = new File(PROJECT);
  if (!aep.exists) GL.fail("AE_PROJECT not found: " + PROJECT);
  try {
    var already = app.project && app.project.file && (app.project.file.fsName === aep.fsName);
    if (!already) app.open(aep);
  } catch(e) { app.open(aep); }

  if (!CSV_PATH) GL.fail("AE_CSV env not set.");
  if (!LEAGUE || GL.cleanValue(LEAGUE) === "") GL.fail("AE_LEAGUE is required.");

  var rows  = GL.parseCSV(GL.openRead(CSV_PATH));
  var teams = GL.buildTeams(rows, LEAGUE);
  var todo  = GL.pickTeamsLeagueOnly(teams, LEAGUE);
  if (LIMIT && todo.length > LIMIT) todo = todo.slice(0, LIMIT);
  if (!todo.length) GL.fail("No teams matched league: " + LEAGUE);

  var comp = GL.findComp(COMP_NAME);
  if (!comp) GL.fail("Comp not found: " + COMP_NAME);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  GL.ensureFolder(rootOut);

  GL.rqClear();

  // --- helper functions ---

  // Set text layer color to primary
  function setTextToPrimary(compItem, layerName, color01) {
    var layer = GL.getLayer(compItem, layerName);
    if (!layer) {
      $.writeln("WARNING: Layer not found: " + layerName);
      return false;
    }
    return GL.setTextOrFillColor(layer, color01);
  }

  // Set layer style stroke color
  function setLayerStyleStroke(layer, color01) {
    if (!layer) return false;
    try {
      var ls = layer.property("ADBE Layer Styles") || layer.property("Layer Styles");
      if (!ls) {
        ls = layer.property("ADBE Layer Styles");
        if (!ls) return false;
      }

      var stroke = ls.property("ADBE Stroke") || ls.property("Stroke");
      if (!stroke) {
        try {
          stroke = ls.addProperty("ADBE Stroke");
        } catch(e) {
          return false;
        }
      }

      if (stroke) {
        var colorProp = stroke.property("ADBE Stroke-0003") || stroke.property("Color");
        if (colorProp) {
          colorProp.setValue(GL.safeColor(color01));
          return true;
        }
      }
    } catch(e) {
      $.writeln("  ✗ Error setting layer style: " + e);
    }
    return false;
  }

  // Fit text to canvas width by adjusting tracking iteratively
  // For center-aligned text: left edge should be at 0, right edge at 64
  function fitTextToCanvas(compItem, textLayer, teamAbbr) {
    if (!textLayer) {
      $.writeln("  ✗ Text layer is null");
      return false;
    }

    try {
      $.writeln("  → Fitting text for: " + teamAbbr);

      // Get source text property
      var sourceProp = textLayer.property("Source Text");
      if (!sourceProp) {
        $.writeln("  ✗ No Source Text property");
        return false;
      }

      // Set text content with tracking 0
      var textDoc = sourceProp.value;
      textDoc.text = String(teamAbbr || "");
      textDoc.tracking = 0;
      sourceProp.setValue(textDoc);

      $.writeln("  → Text set to: '" + textDoc.text + "'");

      var compWidth = compItem.width;
      var targetLeftEdge = 1;
      var targetRightEdge = compWidth;
      var tolerance = 1; // pixels

      $.writeln("  → Target: left edge = " + targetLeftEdge + ", right edge = " + targetRightEdge);

      // Iteratively adjust tracking
      var maxIterations = 20;
      var tracking = 0;

      for (var iter = 0; iter < maxIterations; iter++) {
        // Force AE to update
        compItem.time = compItem.time + 0.001;
        compItem.time = compItem.time - 0.001;

        // Get text bounds
        var rect = null;
        try {
          rect = textLayer.sourceRectAtTime(compItem.time, false);
        } catch(rectError) {
          $.writeln("  ✗ sourceRectAtTime failed: " + rectError.toString());
          break;
        }

        if (!rect || !rect.width || rect.width <= 0) {
          $.writeln("  ✗ Could not get valid text bounds");
          break;
        }

        // Calculate actual edges (center-aligned text)
        var leftEdge = rect.left;
        var rightEdge = rect.left + rect.width;

        $.writeln("  → Iteration " + (iter + 1) + ": tracking=" + tracking +
                  ", left=" + Math.round(leftEdge * 10) / 10 +
                  ", right=" + Math.round(rightEdge * 10) / 10 +
                  ", width=" + Math.round(rect.width * 10) / 10);

        // Check if we're close enough
        var leftError = leftEdge - targetLeftEdge;
        var rightError = rightEdge - targetRightEdge;

        if (Math.abs(leftError) <= tolerance && Math.abs(rightError) <= tolerance) {
          $.writeln("  ✓ Text fitted! Final tracking: " + tracking);
          return true;
        }

        // Calculate how much we need to expand/contract
        // For center-aligned text, we care about the total width
        var currentWidth = rect.width;
        var targetWidth = targetRightEdge - targetLeftEdge;
        var widthDiff = targetWidth - currentWidth;

        if (Math.abs(widthDiff) <= tolerance) {
          $.writeln("  ✓ Text fitted! Final tracking: " + tracking);
          return true;
        }

        // Adjust tracking based on width difference
        // Empirical: tracking change ≈ widthDiff * 10 for Montserrat
        var trackingAdjustment = Math.round(widthDiff * 10);

        // Dampen adjustment to avoid overshooting
        if (iter > 5) {
          trackingAdjustment = Math.round(trackingAdjustment * 0.5);
        }

        tracking += trackingAdjustment;

        // Clamp tracking to reasonable range
        tracking = Math.max(-300, Math.min(500, tracking));

        // Apply new tracking
        textDoc = sourceProp.value;
        textDoc.tracking = tracking;
        sourceProp.setValue(textDoc);
      }

      $.writeln("  ⚠ Max iterations reached, final tracking: " + tracking);
      return true;

    } catch(e) {
      $.writeln("  ✗ Error in fitTextToCanvas: " + e.toString());
      if (e.line) $.writeln("  → Line: " + e.line);

      // Try to at least set the text
      try {
        var sp = textLayer.property("Source Text");
        if (sp) {
          var td = sp.value;
          td.text = String(teamAbbr || "");
          td.tracking = 0;
          sp.setValue(td);
          $.writeln("  → Text set (no tracking adjustment)");
        }
      } catch(e2) {
        $.writeln("  ✗ Could not set text: " + e2.toString());
      }

      return false;
    }
  }

  // Import and replace logo
  function findOrImportFootage(filePath) {
    var f = new File(filePath);
    if (!f.exists) return null;
    var fs = f.fsName;

    // Check if already imported
    for (var i = 1; i <= app.project.numItems; i++) {
      var it = app.project.item(i);
      try {
        if (it instanceof FootageItem && it.file && it.file.fsName === fs) return it;
      } catch(e) {}
    }

    // Import new
    try {
      var io = new ImportOptions(f);
      if (io.canImportAs(ImportAsType.FOOTAGE)) {
        io.importAs = ImportAsType.FOOTAGE;
        return app.project.importFile(io);
      }
    } catch(e) {}
    return null;
  }

  // --- per team loop ---
  $.writeln("\n========================================");
  $.writeln("Processing " + todo.length + " teams for league: " + LEAGUE);
  $.writeln("Comp: " + COMP_NAME);
  $.writeln("Animation: " + ANIM_NAME);
  $.writeln("========================================\n");

  for (var i = 0; i < todo.length; i++) {
    var t = todo[i];
    $.writeln("["+(i+1)+"/"+todo.length+"] Processing: " + t.abbr + " (" + t.name + ")");

    var smart = GL.computeSmartColors(GL.safeColor(t.primary), GL.safeColor(t.secondary));
    var P = smart.primary;
    var S = smart.secondary;

    // 1) Set TOUCHDOWN text to Primary
    if (setTextToPrimary(comp, TOUCHDOWN_LAYER, P)) {
      $.writeln("  ✓ Set TOUCHDOWN to Primary color");
    }

    // 2) Set TOUCHDOWN 2 text to Primary
    if (setTextToPrimary(comp, TOUCHDOWN2_LAYER, P)) {
      $.writeln("  ✓ Set TOUCHDOWN 2 to Primary color");
    }

    // 3) Set TeamName text, fit to canvas, and set layer style to Secondary
    var teamNameLayer = GL.getLayer(comp, TEAMNAME_LAYER);
    if (teamNameLayer) {
      fitTextToCanvas(comp, teamNameLayer, t.abbr);
      if (setLayerStyleStroke(teamNameLayer, S)) {
        $.writeln("  ✓ Set TeamName layer style to Secondary color");
      }
    } else {
      $.writeln("  ✗ TeamName layer not found");
    }

    // 4) Replace logos
    var logoFile = GL.findLogoFile(t.league, t.abbr, logoOpts);
    if (logoFile) {
      $.writeln("  → Found logo: " + logoFile.fsName);
      var footage = findOrImportFootage(logoFile.fsName);

      if (footage) {
        $.writeln("  ✓ Imported logo: " + footage.name);

        // Replace Main-Logo-PreComp (just replace, don't change scale)
        var mainLogoLayer = GL.getLayer(comp, MAIN_LOGO_PRECOMP);
        if (mainLogoLayer && mainLogoLayer.source && (mainLogoLayer.source instanceof CompItem)) {
          var precomp = mainLogoLayer.source;
          var replaced = 0;
          for (var L = 1; L <= precomp.numLayers; L++) {
            var subLayer = precomp.layer(L);
            try {
              if (subLayer.source && (subLayer.source instanceof FootageItem)) {
                subLayer.replaceSource(footage, false);
                $.writeln("  ✓ Replaced '" + subLayer.name + "' in Main-Logo-PreComp");
                replaced++;
              }
            } catch(e) {
              $.writeln("  ✗ Error replacing '" + subLayer.name + "': " + e);
            }
          }
          if (replaced === 0) {
            $.writeln("  ✗ No footage layers in Main-Logo-PreComp");
          }
        }

        // Replace TeamLogo (just replace, don't change scale)
        var teamLogoLayer = GL.getLayer(comp, TEAMLOGO_LAYER);
        if (teamLogoLayer && teamLogoLayer.source) {
          try {
            teamLogoLayer.replaceSource(footage, false);
            $.writeln("  ✓ Replaced TeamLogo");
          } catch(e) {
            $.writeln("  ✗ Error replacing TeamLogo: " + e);
          }
        }
      }
    } else {
      $.writeln("  ✗ Logo file not found for " + t.abbr);
    }

    // Pixel snap
    GL.snapCompTextForPixelArt(comp);

    if (PURGE && app.purge) {
      try { app.purge(PurgeTarget.ALL_CACHES); } catch(e) {}
    }

    // Render
    if (!NO_RENDER) {
      var lc = GL.leagueAndConfForPath(t.league, t.conference);
      var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
      $.writeln("  → Output: " + paths.file.fsName);
      GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
    }

    // Clean up logo - DISABLED FOR TESTING
    // var removed = GL.removeLogoByAbbr(t.abbr);
    // if (removed > 0) $.writeln("  ✓ Removed " + removed + " logo(s)");

    $.writeln("");
  }

  $.writeln("========================================");
  $.writeln("Batch complete: " + todo.length + " teams processed");
  $.writeln("========================================");

  app.endUndoGroup();
  if (app.endSuppressDialogs) { try { app.endSuppressDialogs(); } catch(e) {} }
  if (QUIT_APP) app.quit();
})();
